<?php
/**
 * Public-facing logic for the plugin.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class AffBox_Public
 */
class AffBox_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @var string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @var string $version The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of the plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Render the affiliate product shortcode.
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string The rendered HTML.
	 */
	public function render_affiliate_product( $atts ) {
		$atts = shortcode_atts(
			array(
				'slug'       => '',
				'identifier' => '',
				'class'      => 'affiliate-product',
			),
			$atts,
			'affprod'
		);

		if ( empty( $atts['slug'] ) && empty( $atts['identifier'] ) ) {
			return '<p class="affbox-error">' . esc_html__( 'Product slug or identifier is required', 'affbox' ) . '</p>';
		}

		$args = array(
			'post_type'      => 'affiliate_product',
			'post_status'    => 'publish',
			'posts_per_page' => 1,
		);

		if ( ! empty( $atts['identifier'] ) ) {
			$args['meta_key']   = '_affprod_identifier'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			$args['meta_value'] = $atts['identifier']; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
		} else {
			$args['meta_key']   = '_affprod_slug'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			$args['meta_value'] = $atts['slug']; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
		}

		// Find the product.
		$product = get_posts( $args );

		if ( empty( $product ) ) {
			return '<p class="affbox-error">' . esc_html__( 'Product not found', 'affbox' ) . '</p>';
		}

		$product      = $product[0];
		$product_id   = $product->ID;
		$product_slug = get_post_meta( $product_id, '_affprod_slug', true );

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$force_refresh = isset( $_GET['refresh_product'] ) && $_GET['refresh_product'] === $product_slug && current_user_can( 'manage_options' );

		// Get the saved content.
		$content      = get_post_meta( $product_id, '_affprod_content', true );
		$last_updated = get_post_meta( $product_id, '_affprod_last_updated', true );

		// If content is empty or force refresh is requested and user has permission.
		if ( empty( $content ) || $force_refresh ) {
			try {
				AffBox_Utils::upsert_product_by_slug( $product_slug );
				$content = get_post_meta( $product_id, '_affprod_content', true );
				if ( function_exists( 'wp_cache_delete' ) ) {
					wp_cache_delete( $product_id, 'post_meta' );
				}
			} catch ( Exception $e ) {
				if ( empty( $content ) ) {
					return '<p class="affbox-error">' .
						esc_html__( 'Error loading product content: ', 'affbox' ) .
						esc_html( $e->getMessage() ) .
						'</p>';
				}
			}
		}

		// Verify SRI hash to detect post-storage content tampering.
		$sri_hash = get_post_meta( $product_id, '_affprod_sri_hash', true );
		if ( ! empty( $sri_hash ) && ! AffBox_Utils::verify_sri_hash( $content, $sri_hash ) ) {
			return '<p class="affbox-error">' .
				esc_html__( 'Product content integrity check failed. Please re-sync this product.', 'affbox' ) .
				'</p>';
		}

		return $content;
	}

	/**
	 * Register the Gutenberg blocks.
	 */
	public function register_blocks() {
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		// Register REST API routes.
		add_action( 'rest_api_init', array( $this, 'register_preview_route' ) );
		add_action( 'rest_api_init', array( $this, 'register_refresh_route' ) );

		// Register block editor script.
		wp_register_script(
			'affbox-block-editor',
			AFFBOX_PLUGIN_URL . 'admin/js/block-editor.js',
			array(
				'wp-blocks',
				'wp-i18n',
				'wp-element',
				'wp-components',
				'wp-block-editor',
				'wp-data',
				'wp-api-fetch',
				'wp-url',
				'wp-hooks',
				'wp-date',
				'choices-js',
			),
			$this->version,
			true
		);

		// Register block.
		register_block_type(
			'affbox/affiliate-product',
			array(
				'api_version'     => 2,
				'editor_script'   => 'affbox-block-editor',
				'render_callback' => array( $this, 'render_block' ),
				'attributes'      => array(
					'slug'      => array(
						'type'    => 'string',
						'default' => '',
					),
					'className' => array(
						'type'    => 'string',
						'default' => '',
					),
				),
			)
		);
	}

	/**
	 * Render the Gutenberg block.
	 *
	 * @param array $attributes Block attributes.
	 * @return string The rendered HTML.
	 */
	public function render_block( $attributes ) {
		if ( empty( $attributes['slug'] ) ) {
			return '<p>' . esc_html__( 'Please select a product to display.', 'affbox' ) . '</p>';
		}

		$class = 'wp-block-affbox-affiliate-product';
		if ( ! empty( $attributes['className'] ) ) {
			$class .= ' ' . esc_attr( $attributes['className'] );
		}

		return $this->render_affiliate_product(
			array(
				'slug'  => $attributes['slug'],
				'class' => $class,
			)
		);
	}

	/**
	 * Register public webhook REST routes for connect and receive.
	 */
	public function register_webhook_routes() {
		register_rest_route(
			'affbox/v1',
			'/connect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_connect' ),
				'permission_callback' => '__return_true',
			)
		);

		register_rest_route(
			'affbox/v1',
			'/receive',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_receive' ),
				'permission_callback' => '__return_true',
			)
		);
	}

	/**
	 * Apply rate limiting based on client IP.
	 *
	 * @return WP_Error|null WP_Error if rate limited, null otherwise.
	 */
	private function check_rate_limit() {
		$client_ip  = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '0.0.0.0';
		$rate_key   = 'affbox_rl_' . md5( $client_ip );
		$rate_count = (int) get_transient( $rate_key );
		if ( $rate_count >= 120 ) {
			return new WP_Error( 'rate_limited', 'Rate limit exceeded. Try again later.', array( 'status' => 429 ) );
		}
		set_transient( $rate_key, $rate_count + 1, MINUTE_IN_SECONDS );
		return null;
	}

	/**
	 * Handle the connect webhook (replaces connect.php).
	 *
	 * Receives an API key and one-time secret from the remote app and saves the key.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_connect( $request ) {
		$rate_error = $this->check_rate_limit();
		if ( null !== $rate_error ) {
			return $rate_error;
		}

		$api_key = $request->get_param( 'api_key' );
		$secret  = $request->get_param( 'secret' );

		if ( empty( $api_key ) ) {
			return new WP_Error( 'missing_api_key', 'Missing api_key field', array( 'status' => 400 ) );
		}

		if ( empty( $secret ) ) {
			return new WP_Error( 'missing_secret', 'Missing secret', array( 'status' => 403 ) );
		}

		$api_key = sanitize_text_field( $api_key );
		$secret  = sanitize_text_field( $secret );

		$stored_secret = get_transient( 'affbox_connection_secret' );
		if ( ! $stored_secret || ! hash_equals( $stored_secret, $secret ) ) {
			return new WP_Error( 'invalid_secret', 'Invalid or expired secret', array( 'status' => 403 ) );
		}

		// Delete the secret so it can't be used again.
		delete_transient( 'affbox_connection_secret' );

		// Save API key.
		$settings            = get_option( 'affbox_settings', array() );
		$settings['api_key'] = $api_key;
		$updated             = update_option( 'affbox_settings', $settings );

		$current_settings = get_option( 'affbox_settings' );
		if ( $updated || ( isset( $current_settings['api_key'] ) && $current_settings['api_key'] === $api_key ) ) {
			return new WP_REST_Response( array( 'status' => 'success' ), 200 );
		}

		return new WP_Error( 'save_failed', 'Failed to save API key', array( 'status' => 500 ) );
	}

	/**
	 * Handle the receive webhook (replaces receive.php).
	 *
	 * Receives product updates/deletes from the remote API and schedules processing.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_receive( $request ) {
		$rate_error = $this->check_rate_limit();
		if ( null !== $rate_error ) {
			return $rate_error;
		}

		$data = $request->get_json_params();

		// Verify API key.
		if ( empty( $data['api_key'] ) ) {
			return new WP_Error( 'missing_api_key', 'Missing required field: api_key', array( 'status' => 400 ) );
		}

		$settings      = get_option( 'affbox_settings', array() );
		$valid_api_key = isset( $settings['api_key'] ) ? $settings['api_key'] : '';
		if ( empty( $valid_api_key ) || ! hash_equals( $valid_api_key, $data['api_key'] ) ) {
			return new WP_Error( 'invalid_api_key', 'Invalid API key', array( 'status' => 401 ) );
		}

		// Normalize input - support both single product and multiple products.
		$products = array();

		if ( isset( $data['products'] ) && is_array( $data['products'] ) ) {
			$products = $data['products'];
		} elseif ( isset( $data['boxes'] ) && is_array( $data['boxes'] ) ) {
			// Legacy format: "boxes" array (backwards compatibility).
			$products = $data['boxes'];
		} elseif ( isset( $data['slug'] ) ) {
			// Legacy format: single product.
			$products[] = array(
				'slug'       => $data['slug'],
				'name'       => isset( $data['name'] ) ? $data['name'] : '',
				'type'       => isset( $data['type'] ) ? $data['type'] : 'box',
				'identifier' => isset( $data['identifier'] ) ? $data['identifier'] : '',
				'deleted'    => isset( $data['deleted'] ) ? $data['deleted'] : false,
			);
		} else {
			return new WP_Error( 'missing_products', 'Missing required field: products or slug', array( 'status' => 400 ) );
		}

		if ( empty( $products ) ) {
			return new WP_Error( 'no_products', 'No products provided', array( 'status' => 400 ) );
		}

		// Process each product.
		$scheduled         = array();
		$processing_errors = array();

		foreach ( $products as $product ) {
			if ( empty( $product['slug'] ) ) {
				$processing_errors[] = 'Product missing required field: slug';
				continue;
			}

			$product_slug  = sanitize_text_field( $product['slug'] );
			$product_title = isset( $product['name'] ) ? sanitize_text_field( $product['name'] ) : '';
			$product_type  = isset( $product['type'] ) ? sanitize_text_field( $product['type'] ) : 'box';

			// Infer type from slug if default 'box' is used.
			if ( 'box' === $product_type && 0 === strpos( $product_slug, 'table_' ) ) {
				$product_type = 'table';
			}

			$product_identifier = isset( $product['identifier'] ) ? sanitize_text_field( $product['identifier'] ) : '';
			$deleted            = isset( $product['deleted'] ) ? (bool) $product['deleted'] : false;

			try {
				if ( $deleted ) {
					$action_id   = AffBox_Action_Scheduler::schedule_product_delete( $product_slug );
					$scheduled[] = array(
						'slug'      => $product_slug,
						'action'    => 'delete',
						'action_id' => $action_id,
					);
				} else {
					$action_id   = AffBox_Action_Scheduler::schedule_product_update( $product_slug, $product_title, $product_type, $product_identifier );
					$scheduled[] = array(
						'slug'      => $product_slug,
						'action'    => 'update',
						'action_id' => $action_id,
					);
				}
			} catch ( Exception $e ) {
				$processing_errors[] = sprintf(
					'Failed to schedule product %s: %s',
					$product_slug,
					$e->getMessage()
				);
			}
		}

		// Force run pending actions immediately.
		if ( ! empty( $scheduled ) ) {
			AffBox_Action_Scheduler::force_run_queue();
		}

		// Prepare response.
		$response = array(
			'success'      => true,
			'message'      => sprintf(
				'Scheduled %d product(s) successfully',
				count( $scheduled )
			),
			'scheduled'    => $scheduled,
			'scheduled_at' => current_time( 'mysql' ),
		);

		if ( ! empty( $processing_errors ) ) {
			$response['errors']     = $processing_errors;
			$response['has_errors'] = true;
		}

		if ( empty( $scheduled ) ) {
			return new WP_Error( 'schedule_failed', 'Failed to schedule any products', array( 'status' => 500 ) );
		}

		return new WP_REST_Response( $response, 200 );
	}

	/**
	 * Register REST API route for preview.
	 */
	public function register_preview_route() {
		register_rest_route(
			'affbox/v1',
			'/preview',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_product_preview' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'slug' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);
	}

	/**
	 * Register REST API route to force refresh a product's stored content.
	 */
	public function register_refresh_route() {
		register_rest_route(
			'affbox/v1',
			'/refresh',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'refresh_product' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'slug' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);
	}

	/**
	 * Refresh a product by slug using utility, return updated preview.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_Error|array The updated preview or error.
	 */
	public function refresh_product( $request ) {
		$slug = $request->get_param( 'slug' );
		if ( empty( $slug ) ) {
			return new WP_Error( 'no_slug', __( 'No slug provided', 'affbox' ), array( 'status' => 400 ) );
		}
		try {
			$post_id = AffBox_Utils::upsert_product_by_slug( $slug );
		} catch ( Exception $e ) {
			return new WP_Error( 'refresh_failed', $e->getMessage(), array( 'status' => 500 ) );
		}
		$html         = $this->render_affiliate_product(
			array(
				'slug'  => $slug,
				'class' => 'affbox-preview',
			)
		);
		$last_updated = get_post_meta( $post_id, '_affprod_last_synced', true );
		return array(
			'success'      => true,
			'post_id'      => $post_id,
			'html'         => $html,
			'slug'         => $slug,
			'last_updated' => $last_updated,
		);
	}

	/**
	 * Get product preview for REST API.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_Error|array The preview or error.
	 */
	public function get_product_preview( $request ) {
		$slug = $request->get_param( 'slug' );

		if ( empty( $slug ) ) {
			return new WP_Error( 'no_slug', __( 'No slug provided', 'affbox' ), array( 'status' => 400 ) );
		}

		$html = $this->render_affiliate_product(
			array(
				'slug'  => $slug,
				'class' => 'affbox-preview',
			)
		);

		// include last updated meta if present.
		$post         = get_posts(
			array(
				'post_type'      => 'affiliate_product',
				'meta_key'       => '_affprod_slug', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				'meta_value'     => $slug, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				'post_status'    => 'any',
				'posts_per_page' => 1,
			)
		);
		$last_updated = '';
		if ( ! empty( $post ) ) {
			$last_updated = get_post_meta( $post[0]->ID, '_affprod_last_synced', true );
		}
		return array(
			'html'         => $html,
			'slug'         => $slug,
			'last_updated' => $last_updated,
		);
	}
}
