<?php
/**
 * The core plugin class.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

/**
 * The core plugin class.
 */
class AffBox {

	/**
	 * The unique identifier of this plugin.
	 *
	 * @var string $plugin_name The ID of this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @var string $version The current version of the plugin.
	 */
	protected $version;

	/**
	 * The admin-specific functionality of the plugin.
	 *
	 * @var AffBox_Admin $admin The admin-specific functionality of the plugin.
	 */
	protected $admin;

	/**
	 * The public-facing functionality of the plugin.
	 *
	 * @var AffBox_Public $public The public-facing functionality of the plugin.
	 */
	protected $public;

	/**
	 * Initialize the class and set its properties.
	 */
	public function __construct() {
		$this->plugin_name = 'affbox';
		$this->version     = defined( 'AFFBOX_VERSION' ) ? AFFBOX_VERSION : '1.0.0';

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->register_post_types();
		$this->init_action_scheduler();
	}

	/**
	 * Initialize Action Scheduler.
	 */
	private function init_action_scheduler() {
		if ( class_exists( 'AffBox_Action_Scheduler' ) ) {
			$action_scheduler = new AffBox_Action_Scheduler();
			$action_scheduler->init();
		}
	}

	/**
	 * Load the required dependencies for the plugin.
	 */
	private function load_dependencies() {
		// The class responsible for admin-specific functionality.
		require_once AFFBOX_PLUGIN_DIR . 'admin/class-affbox-admin.php';

		// The class responsible for public-facing functionality.
		require_once AFFBOX_PLUGIN_DIR . 'public/class-affbox-public.php';

		// Initialize admin and public classes.
		$this->admin  = new AffBox_Admin( $this->get_plugin_name(), $this->get_version() );
		$this->public = new AffBox_Public( $this->get_plugin_name(), $this->get_version() );
	}

	/**
	 * Register all of the hooks related to the admin area functionality.
	 */
	private function define_admin_hooks() {
		// Add admin menu.
		add_action( 'admin_menu', array( $this->admin, 'add_plugin_admin_menu' ) );

		// Enqueue admin scripts and styles.
		add_action( 'admin_enqueue_scripts', array( $this->admin, 'enqueue_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this->admin, 'enqueue_scripts' ) );

		// AJAX handlers.
		add_action( 'wp_ajax_affbox_sync_products', array( $this->admin, 'sync_products' ) );
		add_action( 'wp_ajax_affbox_get_sync_status', array( $this->admin, 'get_sync_status' ) );
		add_action( 'wp_ajax_affbox_get_queue_status', array( $this->admin, 'get_queue_status' ) );
		add_action( 'wp_ajax_affbox_disconnect_site', array( $this->admin, 'disconnect_site' ) );
		add_action( 'wp_ajax_affbox_get_available_products', array( $this->admin, 'get_available_products' ) );
		add_action( 'wp_ajax_affbox_start_connection', array( $this->admin, 'start_connection' ) );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality.
	 */
	private function define_public_hooks() {
		// Enqueue block editor assets.
		add_action( 'enqueue_block_editor_assets', array( $this->admin, 'enqueue_block_editor_assets' ) );

		// Register shortcodes.
		add_shortcode( 'affprod', array( $this->public, 'render_affiliate_product' ) );
		add_shortcode( 'afftable', array( $this->public, 'render_affiliate_product' ) );
		add_shortcode( 'affbox', array( $this->public, 'render_affiliate_product' ) ); // Legacy alias.

		// Register Gutenberg block.
		add_action( 'init', array( $this->public, 'register_blocks' ) );

		// Register public webhook REST routes (connect + receive).
		add_action( 'rest_api_init', array( $this->public, 'register_webhook_routes' ) );
	}

	/**
	 * Register custom post types and taxonomies.
	 */
	private function register_post_types() {
		add_action( 'init', array( $this, 'register_affiliate_product_post_type' ) );
		add_action( 'init', array( $this, 'register_affiliate_product_meta' ) );
	}

	/**
	 * Register meta fields for REST API access.
	 */
	public static function register_affiliate_product_meta() {
		register_post_meta(
			'affiliate_product',
			'_affprod_slug',
			array(
				'show_in_rest'  => true,
				'single'        => true,
				'type'          => 'string',
				'auth_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);

		register_post_meta(
			'affiliate_product',
			'_affprod_content',
			array(
				'show_in_rest'  => array(
					'schema' => array(
						'type'     => 'string',
						'readonly' => true,
					),
				),
				'single'        => true,
				'type'          => 'string',
				'auth_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);

		register_post_meta(
			'affiliate_product',
			'_affprod_last_synced',
			array(
				'show_in_rest'  => true,
				'single'        => true,
				'type'          => 'string',
				'auth_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);

		register_post_meta(
			'affiliate_product',
			'_affprod_identifier',
			array(
				'show_in_rest'  => true,
				'single'        => true,
				'type'          => 'string',
				'auth_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);

		register_post_meta(
			'affiliate_product',
			'_affprod_type',
			array(
				'show_in_rest'  => true,
				'single'        => true,
				'type'          => 'string',
				'auth_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);
	}


	/**
	 * Run the plugin logic.
	 */
	public function run() {
		// No loader needed, hooks are already registered in the constructor.
	}


	/**
	 * Get the plugin name.
	 *
	 * @return string The plugin name.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Get the version of the plugin.
	 *
	 * @return string The version.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Activation hook.
	 */
	public static function activate() {
		// Register post type and flush rewrite rules.
		self::register_affiliate_product_post_type();

		// Run migration from old post type/meta keys if needed.
		self::migrate_from_boxes();

		// Set default options if they don't exist.
		if ( false === get_option( 'affbox_settings' ) ) {
			update_option(
				'affbox_settings',
				array(
					'api_key' => '',
				)
			);
		}
	}

	/**
	 * Migrate from old affiliate_box post type and _affbox_* meta keys.
	 */
	private static function migrate_from_boxes() {
		// Check if migration has already been done.
		if ( get_option( 'affbox_migrated_to_products' ) ) {
			return;
		}

		global $wpdb;

		// Get all posts with old post type.
		$old_posts = get_posts(
			array(
				'post_type'      => 'affiliate_box',
				'post_status'    => 'any',
				'posts_per_page' => -1,
				'fields'         => 'ids',
			)
		);

		if ( ! empty( $old_posts ) ) {
			foreach ( $old_posts as $post_id ) {
				// Update post type.
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->update(
					$wpdb->posts,
					array( 'post_type' => 'affiliate_product' ),
					array( 'ID' => $post_id ),
					array( '%s' ),
					array( '%d' )
				);

				// Migrate meta keys.
				$meta_mappings = array(
					'_affbox_slug'        => '_affprod_slug',
					'_affbox_content'     => '_affprod_content',
					'_affbox_last_synced' => '_affprod_last_synced',
					'_affbox_identifier'  => '_affprod_identifier',
					'_affbox_type'        => '_affprod_type',
				);

				foreach ( $meta_mappings as $old_key => $new_key ) {
					$value = get_post_meta( $post_id, $old_key, true );
					if ( '' !== $value ) {
						update_post_meta( $post_id, $new_key, $value );
						delete_post_meta( $post_id, $old_key );
					}
				}
			}
		}

		// Mark migration as complete.
		update_option( 'affbox_migrated_to_products', true );

		// Clear cache.
		if ( function_exists( 'wp_cache_flush' ) ) {
			wp_cache_flush();
		}
	}

	/**
	 * Deactivation hook.
	 */
	public static function deactivate() {
		// nothing to do here.
	}

	/**
	 * Register the Affiliate Product post type.
	 */
	public static function register_affiliate_product_post_type() {
		$args = array(
			'label'               => __( 'Affiliate Products', 'affbox' ),
			'labels'              => array(
				'name'          => __( 'Affiliate Products', 'affbox' ),
				'singular_name' => __( 'Affiliate Product', 'affbox' ),
				'menu_name'     => __( 'Affiliate Products', 'affbox' ),
			),
			'public'              => false,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'supports'            => array( 'title', 'custom-fields' ),
			'publicly_queryable'  => false,
			'show_in_admin_bar'   => false,
			'show_in_nav_menus'   => false,
			'can_export'          => false,
			'has_archive'         => false,
			'exclude_from_search' => true,
			'show_in_rest'        => true,
		);
		register_post_type( 'affiliate_product', $args );
	}
}
