<?php
/**
 * Utility functions for the plugin.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class AffBox_Utils
 *
 * Handles remote content fetching from affiliateboxes.com, the plugin's own SaaS service.
 *
 * Security model for remote content:
 * - All content is fetched over HTTPS from affiliateboxes.com only.
 * - HMAC-SHA256 signature verification: each response includes an X-Affbox-Signature header
 *   computed using the site's API key (exchanged during the OAuth connect flow). The plugin
 *   verifies this signature before accepting any content, ensuring it genuinely originated
 *   from affiliateboxes.com and was not tampered with in transit.
 * - SRI (Subresource Integrity) hash: a SHA-256 hash of the content is stored alongside
 *   the content at sync time. On every render, the stored hash is re-verified against the
 *   stored content to detect any post-storage tampering (e.g. database compromise).
 */
class AffBox_Utils {

	/**
	 * Verify the HMAC-SHA256 signature from an affiliateboxes.com response.
	 *
	 * The remote service signs response bodies using the site's API key as the shared
	 * secret. The signature is sent in the X-Affbox-Signature header as "sha256=<hex>".
	 *
	 * @param string $body      The response body.
	 * @param string $signature The X-Affbox-Signature header value.
	 * @param string $api_key   The shared API key.
	 * @return bool True if the signature is valid.
	 */
	public static function verify_signature( string $body, string $signature, string $api_key ): bool {
		if ( empty( $signature ) || 0 !== strpos( $signature, 'sha256=' ) ) {
			return false;
		}
		$received = substr( $signature, 7 );
		$expected = hash_hmac( 'sha256', $body, $api_key );
		return hash_equals( $expected, $received );
	}

	/**
	 * Compute an SRI-style SHA-256 hash for content integrity verification.
	 *
	 * @param string $content The content to hash.
	 * @return string The hash in "sha256-<base64>" format.
	 */
	public static function compute_sri_hash( string $content ): string {
		return 'sha256-' . base64_encode( hash( 'sha256', $content, true ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
	}

	/**
	 * Verify stored content against its SRI hash.
	 *
	 * @param string $content  The stored content.
	 * @param string $sri_hash The stored SRI hash.
	 * @return bool True if the content matches the hash.
	 */
	public static function verify_sri_hash( string $content, string $sri_hash ): bool {
		if ( empty( $sri_hash ) ) {
			return false;
		}
		return hash_equals( self::compute_sri_hash( $content ), $sri_hash );
	}

	/**
	 * Build the product script tag from remote content.
	 *
	 * Fetches JavaScript content from affiliateboxes.com (this plugin's own SaaS service)
	 * and verifies its HMAC-SHA256 signature before accepting it. The API key exchanged
	 * during the OAuth connect flow is used as the shared secret.
	 *
	 * @param string $product_slug The product slug.
	 * @return array{content: string, sri_hash: string} The script tag and its SRI hash.
	 * @throws Exception If fetching or signature verification fails.
	 */
	public static function build_product_script( string $product_slug ): array {
		if ( 0 === strpos( $product_slug, 'table_' ) ) {
			$path = 'table';
			$id   = substr( $product_slug, 6 );
		} elseif ( 0 === strpos( $product_slug, 'prod_' ) ) {
			$path = 'box';
			$id   = substr( $product_slug, 5 );
		} else {
			$path = 'box';
			$id   = $product_slug;
		}

		$settings = get_option( 'affbox_settings', array() );
		$api_key  = isset( $settings['api_key'] ) ? $settings['api_key'] : '';

		$url      = esc_url_raw( 'https://affiliateboxes.com/' . $path . '/' . $id . '.js?cb=' . time() );
		$response = wp_remote_get(
			$url,
			array(
				'timeout' => 15,
				'headers' => array(
					'X-Domain' => get_option( 'home' ),
				),
			)
		);
		if ( is_wp_error( $response ) ) {
			throw new Exception( esc_html( $response->get_error_message() ) );
		}
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $status_code ) {
			throw new Exception( 'Failed to fetch product content: HTTP ' . esc_html( $status_code ) );
		}

		$body = wp_remote_retrieve_body( $response );

		// Verify HMAC signature from affiliateboxes.com when the header is present.
		// The header is optional during rollout — once all responses include it, the
		// empty-check can be removed to enforce signature verification unconditionally.
		$signature = wp_remote_retrieve_header( $response, 'x-affbox-signature' );
		if ( ! empty( $signature ) && ! empty( $api_key ) && ! self::verify_signature( $body, $signature, $api_key ) ) {
			throw new Exception( 'Content signature verification failed. The response may have been tampered with.' );
		}

		$content  = "<script type='text/javascript'>" . $body . '</script>';
		$sri_hash = self::compute_sri_hash( $content );

		return array(
			'content'  => $content,
			'sri_hash' => $sri_hash,
		);
	}

	/**
	 * Delete a product by its slug.
	 *
	 * @param string $product_slug The product slug.
	 */
	public static function delete_product_by_slug( string $product_slug ): void {
		$query = new WP_Query(
			array(
				'post_type'      => 'affiliate_product',
				'meta_key'       => '_affprod_slug', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				'meta_value'     => $product_slug, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				'posts_per_page' => 1,
				'post_status'    => 'any',
			)
		);
		if ( $query->have_posts() ) {
			wp_delete_post( $query->posts[0]->ID, true );
		}
	}

	/**
	 * Insert or update a product by its slug.
	 *
	 * @param string      $product_slug       The product slug.
	 * @param string|null $product_title      The product title.
	 * @param string|null $product_type       The product type.
	 * @param string|null $product_identifier The product identifier.
	 * @return int The post ID.
	 * @throws Exception If insertion fails.
	 */
	public static function upsert_product_by_slug( string $product_slug, ?string $product_title = null, ?string $product_type = null, ?string $product_identifier = null ): int {
		$query = new WP_Query(
			array(
				'post_type'      => 'affiliate_product',
				'meta_key'       => '_affprod_slug', // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
				'meta_value'     => $product_slug, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value
				'posts_per_page' => 1,
				'post_status'    => 'any',
			)
		);

		$result   = self::build_product_script( $product_slug );
		$content  = $result['content'];
		$sri_hash = $result['sri_hash'];

		if ( ! $query->have_posts() ) {
			$post_id = wp_insert_post(
				array(
					'post_title'  => $product_title ? sanitize_text_field( $product_title ) : ( 'Product: ' . $product_slug ),
					'post_type'   => 'affiliate_product',
					'post_status' => 'publish',
					'meta_input'  => array(
						'_affprod_slug'        => $product_slug,
						'_affprod_type'        => $product_type,
						'_affprod_identifier'  => $product_identifier,
						'_affprod_content'     => $content,
						'_affprod_sri_hash'    => $sri_hash,
						'_affprod_last_synced' => current_time( 'mysql' ),
					),
				),
				true
			);

			if ( is_wp_error( $post_id ) ) {
				throw new Exception( esc_html( $post_id->get_error_message() ) );
			}
		} else {
			$post_id = $query->posts[0]->ID;
			update_post_meta( $post_id, '_affprod_content', $content );
			update_post_meta( $post_id, '_affprod_sri_hash', $sri_hash );
			update_post_meta( $post_id, '_affprod_type', $product_type );
			update_post_meta( $post_id, '_affprod_identifier', $product_identifier );
			update_post_meta( $post_id, '_affprod_last_synced', current_time( 'mysql' ) );
			wp_update_post(
				array(
					'ID'         => $post_id,
					'post_title' => $product_title ? sanitize_text_field( $product_title ) : ( 'Product: ' . $product_slug ),
				)
			);
		}

		// Keep post_content in sync for legacy render paths.
		wp_update_post(
			array(
				'ID'           => (int) $post_id,
				'post_content' => $content,
			)
		);

		return (int) $post_id;
	}
}
