<?php
/**
 * Logger for AffBox plugin.
 *
 * Writes to a protected log file in wp-content/affbox-logs/.
 * Only logs when WP_DEBUG is enabled.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

class AffBox_Logger {

	/**
	 * Log a message to the plugin log file.
	 *
	 * @param string $message The message to log.
	 */
	public static function log( $message ) {
		if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
			return;
		}

		$log_dir = WP_CONTENT_DIR . '/affbox-logs';

		if ( ! is_dir( $log_dir ) ) {
			wp_mkdir_p( $log_dir );
			self::protect_log_directory( $log_dir );
		}

		$log_file = $log_dir . '/affbox-' . gmdate( 'Y-m-d' ) . '-' . wp_hash( 'affbox-log' ) . '.log';

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		error_log(
			sprintf( '[%s] %s', current_time( 'mysql' ), $message ) . PHP_EOL,
			3,
			$log_file
		);
	}

	/**
	 * Write protective files to prevent direct access to log directory.
	 *
	 * @param string $log_dir The log directory path.
	 */
	private static function protect_log_directory( $log_dir ) {
		// .htaccess for Apache.
		$htaccess = $log_dir . '/.htaccess';
		if ( ! file_exists( $htaccess ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			file_put_contents( $htaccess, 'deny from all' );
		}

		// index.php to prevent directory listing.
		$index = $log_dir . '/index.php';
		if ( ! file_exists( $index ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			file_put_contents( $index, '<?php // Silence is golden.' );
		}
	}
}
