<?php
/**
 * Action Scheduler Handler for AffBox.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

class AffBox_Action_Scheduler {

	public function init() {
		add_action( 'affbox_update_product', array( $this, 'handle_product_update' ), 10, 4 );
		add_action( 'affbox_delete_product', array( $this, 'handle_product_delete' ), 10, 1 );
	}

	public function handle_product_update( $product_slug, $product_title = null, $product_type = null, $product_identifier = null ) {
		$product_slug = sanitize_text_field( $product_slug );

		try {
			$post_id = AffBox_Utils::upsert_product_by_slug( $product_slug, $product_title, $product_type, $product_identifier );
		} catch ( Exception $e ) {
			AffBox_Logger::log( sprintf( 'Failed to update product %s: %s', $product_slug, $e->getMessage() ) );
			throw $e;
		}

		AffBox_Logger::log( sprintf( 'Successfully updated product %s (Post ID: %d)', $product_slug, $post_id ) );

		do_action( 'affbox_product_updated', $post_id, $product_slug, get_post_meta( $post_id, '_affprod_content', true ) );
	}

	public function handle_product_delete( $product_slug ) {
		$product_slug = sanitize_text_field( $product_slug );

		AffBox_Utils::delete_product_by_slug( $product_slug );

		AffBox_Logger::log( sprintf( 'Successfully deleted product %s', $product_slug ) );

		do_action( 'affbox_product_deleted', $product_slug );
	}

	private static function cancel_pending_actions( $product_slug, $hook ) {
		global $wpdb;

		$hook         = sanitize_key( $hook );
		$product_slug = sanitize_text_field( $product_slug );

		$table_actions = $wpdb->actionscheduler_actions;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot use placeholders.
		$action_ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT action_id
				 FROM {$table_actions}
				 WHERE hook = %s
				 AND args LIKE %s
				 AND status IN ('pending','in-progress')",
				$hook,
				'%' . $wpdb->esc_like( '"' . $product_slug . '"' ) . '%'
			)
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		foreach ( $action_ids as $action_id ) {

			as_unschedule_action( $hook, null, 'affbox' );

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->update(
				$table_actions,
				array( 'status' => 'canceled' ),
				array( 'action_id' => (int) $action_id ),
				array( '%s' ),
				array( '%d' )
			);
		}

		if ( ! empty( $action_ids ) ) {
			AffBox_Logger::log(
				sprintf(
					'Canceled %d pending action(s) for product %s',
					count( $action_ids ),
					$product_slug
				)
			);
		}
	}

	public static function schedule_product_update( $product_slug, $product_title = null, $product_type = null, $product_identifier = null ) {
		$product_slug = sanitize_text_field( $product_slug );

		self::cancel_pending_actions( $product_slug, 'affbox_update_product' );

		return as_schedule_single_action(
			time(),
			'affbox_update_product',
			array( $product_slug, $product_title, $product_type, $product_identifier ),
			'affbox'
		);
	}

	public static function schedule_product_delete( $product_slug ) {
		$product_slug = sanitize_text_field( $product_slug );

		self::cancel_pending_actions( $product_slug, 'affbox_delete_product' );

		return as_schedule_single_action(
			time(),
			'affbox_delete_product',
			array( $product_slug ),
			'affbox'
		);
	}

	public static function schedule_bulk_sync( $products ) {
		$batch_id        = 'sync_' . time() . '_' . wp_generate_password( 8, false );
		$scheduled_count = 0;

		foreach ( $products as $product ) {

			if ( empty( $product['slug'] ) ) {
				continue;
			}

			$product_slug       = sanitize_text_field( $product['slug'] );
			$product_title      = $product['name'] ?? null;
			$product_type       = $product['type'] ?? null;
			$product_identifier = $product['identifier'] ?? null;

			as_schedule_single_action(
				time(),
				'affbox_update_product',
				array( $product_slug, $product_title, $product_type, $product_identifier, $batch_id ),
				'affbox'
			);

			++$scheduled_count;
		}

		set_transient(
			'affbox_sync_batch_' . $batch_id,
			array(
				'total'      => $scheduled_count,
				'start_time' => current_time( 'mysql' ),
				'products'   => array_column( $products, 'slug' ),
			),
			HOUR_IN_SECONDS
		);

		return $batch_id;
	}

	public static function get_sync_queue_status( $batch_id ) {
		global $wpdb;

		$batch_id = sanitize_text_field( $batch_id );

		$batch_data = get_transient( 'affbox_sync_batch_' . $batch_id );

		if ( ! $batch_data ) {
			return array(
				'error'    => 'Batch not found or expired',
				'batch_id' => $batch_id,
			);
		}

		$table_actions = $wpdb->actionscheduler_actions;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot use placeholders.
		$pending = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook = %s
				 AND args LIKE %s
				 AND status IN ('pending','in-progress')",
				'affbox_update_product',
				'%' . $wpdb->esc_like( $batch_id ) . '%'
			)
		);

		$completed = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook = %s
				 AND args LIKE %s
				 AND status = 'complete'",
				'affbox_update_product',
				'%' . $wpdb->esc_like( $batch_id ) . '%'
			)
		);

		$failed = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook = %s
				 AND args LIKE %s
				 AND status = 'failed'",
				'affbox_update_product',
				'%' . $wpdb->esc_like( $batch_id ) . '%'
			)
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$is_complete = ( 0 === $pending );

		if ( $is_complete ) {
			delete_transient( 'affbox_sync_batch_' . $batch_id );
		}

		return array(
			'batch_id'    => $batch_id,
			'total'       => (int) $batch_data['total'],
			'completed'   => $completed,
			'pending'     => $pending,
			'failed'      => $failed,
			'is_complete' => $is_complete,
			'start_time'  => $batch_data['start_time'],
		);
	}

	/**
	 * Force run pending actions in the queue.
	 */
	public static function force_run_queue() {
		if ( class_exists( 'ActionScheduler_QueueRunner' ) ) {
			ActionScheduler_QueueRunner::instance()->run();
		}
	}

	/**
	 * Get overall queue status (not batch-specific).
	 */
	public static function get_overall_queue_status() {
		global $wpdb;

		$table_actions = $wpdb->actionscheduler_actions;

		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot use placeholders.
		$pending = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook IN (%s, %s)
				 AND status IN ('pending','in-progress')",
				'affbox_update_product',
				'affbox_delete_product'
			)
		);

		$completed = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook IN (%s, %s)
				 AND status = 'complete'",
				'affbox_update_product',
				'affbox_delete_product'
			)
		);

		$failed = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table_actions}
				 WHERE hook IN (%s, %s)
				 AND status = 'failed'",
				'affbox_update_product',
				'affbox_delete_product'
			)
		);
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		return array(
			'pending'     => $pending,
			'completed'   => $completed,
			'failed'      => $failed,
			'is_complete' => ( 0 === $pending ),
		);
	}
}
