/**
 * Block editor integration for AffBox.
 *
 * @package affbox
 */

const { __ } = wp.i18n;
const { registerBlockType } = wp.blocks;
const { SelectControl, Placeholder, Spinner, PanelBody, ToolbarGroup, ToolbarButton } = wp.components;
const { withSelect } = wp.data;
const { InspectorControls, BlockControls, useBlockProps } = wp.blockEditor;
const { Component, useState, useEffect, useRef } = wp.element;
const el = wp.element.createElement;
const { date } = wp;

// Choices.js wrapper component
const ChoicesSelect = ({ label, value, options, onChange }) => {
	const selectRef = useRef(null);
	const choicesRef = useRef(null);
	const prevOptionsRef = useRef(null);

	useEffect(() => {
		if (selectRef.current) {
			choicesRef.current = new Choices(selectRef.current, {
				searchEnabled: true,
				itemSelectText: '',
				shouldSort: false,
			});

			// Set initial choices and value
			choicesRef.current.setChoices(options, 'value', 'label', true);
			if (value) {
				choicesRef.current.setChoiceByValue(value);
			}

			selectRef.current.addEventListener('change', (e) => {
				onChange(e.target.value);
			});

			prevOptionsRef.current = options;
		}

		return () => {
			if (choicesRef.current) {
				choicesRef.current.destroy();
			}
		};
	}, []);

	// Update choices only if options actually change
	useEffect(() => {
		if (choicesRef.current && prevOptionsRef.current !== options) {
			choicesRef.current.setChoices(options, 'value', 'label', true);
			if (value) {
				choicesRef.current.setChoiceByValue(value);
			}
			prevOptionsRef.current = options;
		}
	}, [options, value]);

	return el(
		'div',
		{ className: 'components-base-control' },
		el(
			'div',
			{ className: 'components-base-control__field' },
			label && el('label', { className: 'components-base-control__label' }, label),
			el(
				'select',
				{ ref: selectRef }
				// Don't render initial options - Choices.js will populate them
			)
		)
	);
};

// Inline SVG icon matching the admin menu icon
//const { createElement: el } = wp.element;

const AffBoxIcon = el(
	'svg',
	{ xmlns: 'http://www.w3.org/2000/svg', width: 20, height: 20, viewBox: '0 0 20 20' },
	el('path', { d: 'M0 0 C1.32 0 2.64 0 4 0 C4 0.66 4 1.32 4 2 C7.63 2 11.26 2 15 2 C15 7.28 15 12.56 15 18 C9.39 18 3.78 18 -2 18 C-2.04125 15.40125 -2.0825 12.8025 -2.125 10.125 C-2.14320801 9.30789551 -2.16141602 8.49079102 -2.18017578 7.64892578 C-2.19477217 5.76504951 -2.09943482 3.88130688 -2 2 C-1.34 1.34 -0.68 0.68 0 0 Z M4 7 C4 8.98 4 10.96 4 13 C5.65 13 7.3 13 9 13 C9 11.02 9 9.04 9 7 C7.35 7 5.7 7 4 7 Z', fill: '#3FBEC7', transform: 'translate(3,0)' }),
	el('path', { d: 'M0 0 C1.98 0 3.96 0 6 0 C6 3.3 6 6.6 6 10 C0.39 10 -5.22 10 -11 10 C-11 8.35 -11 6.7 -11 5 C-7.37 5 -3.74 5 0 5 C0 3.35 0 1.7 0 0 Z', fill: '#FEC91E', transform: 'translate(12,8)' }),
	el('path', { d: 'M0 0 C1.98 0 3.96 0 6 0 C6 0.66 6 1.32 6 2 C4.68 2 3.36 2 2 2 C1.67 3.65 1.34 5.3 1 7 C-1.31 6.67 -3.62 6.34 -6 6 C-6.33 7.32 -6.66 8.64 -7 10 C-8.32 10 -9.64 10 -11 10 C-11 8.35 -11 6.7 -11 5 C-7.37 5 -3.74 5 0 5 C0 3.35 0 1.7 0 0 Z', fill: '#F39238', transform: 'translate(12,8)' }),
	el('path', { d: 'M0 0 C1.98 0 3.96 0 6 0 C6 1.98 6 3.96 6 6 C4.02 6 2.04 6 0 6 C0 4.02 0 2.04 0 0 Z', fill: '#D082FB', transform: 'translate(1,7)' }),
	el('path', { d: 'M0 0 C1.65 0 3.3 0 5 0 C5 1.98 5 3.96 5 6 C3.02 6 1.04 6 -1 6 C-0.67 4.02 -0.34 2.04 0 0 Z', fill: '#E5CDF5', transform: 'translate(13,2)' }),
	el('path', { d: 'M0 0 C1.32 0 2.64 0 4 0 C3.67 1.98 3.34 3.96 3 6 C1.35 6.33 -0.3 6.66 -2 7 C-2.125 4.625 -2.125 4.625 -2 2 C-1.34 1.34 -0.68 0.68 0 0 Z', fill: '#118BB6', transform: 'translate(3,0)' }),
	el('path', { d: 'M0 0 C1.98 0 3.96 0 6 0 C6 1.32 6 2.64 6 4 C3.69 4 1.38 4 -1 4 C-0.67 2.68 -0.34 1.36 0 0 Z', fill: '#FDB213', transform: 'translate(6,14)' }),
	el('path', { d: 'M0 0 C1.98 0 3.96 0 6 0 C5.34 1.65 4.68 3.3 4 5 C2.68 5 1.36 5 0 5 C0 3.35 0 1.7 0 0 Z', fill: '#C33643', transform: 'translate(1,13)' }),
	el('path', { d: 'M0 0 C0.99 0 1.98 0 3 0 C3 1.65 3 3.3 3 5 C1.68 5 0.36 5 -1 5 C-0.67 3.35 -0.34 1.7 0 0 Z', fill: '#91C7D1', transform: 'translate(10,2)' })
);



class ProductPreview extends Component {
	constructor(props) {
		super(props);
		this.state = {
			loading: true,
			html: ''
		};
		this.previewRef = wp.element.createRef();
	}

	componentDidMount() {
		this.fetchPreview();
	}

	componentDidUpdate(prevProps) {
		const slugChanged = prevProps.slug !== this.props.slug && this.props.slug;
		const refreshChanged = prevProps.refreshKey !== this.props.refreshKey;
		if (slugChanged) {
			this.setState({ loading: true, html: '' });
			this.fetchPreview(false);
		} else if (refreshChanged) {
			// Silent refresh: keep showing current HTML and let editor gray it out
			this.fetchPreview(true);
		}
	}

	fetchPreview(silent) {
		const { slug } = this.props;

		if (!slug) {
			this.setState({ loading: false, html: '' });
			return;
		}

		if (!silent) {
			this.setState({ loading: true });
		}
		wp.apiFetch({
			path: '/affbox/v1/preview?slug=' + encodeURIComponent(slug)
		}).then((response) => {
			this.setState({ loading: false, html: response.html || '' }, () => this.renderContent());
			if (this.props.onPreviewLoaded && response && typeof response.last_updated !== 'undefined') {
				this.props.onPreviewLoaded(response.last_updated || '');
			}
		}).catch((error) => {
			this.setState({
				loading: false,
				html: ''
			});
		});
	}

	renderContent() {
		if (this.previewRef.current && this.state.html) {
			this.previewRef.current.innerHTML = '';
			this.previewRef.current.innerHTML = this.state.html;
			const scripts = this.previewRef.current.querySelectorAll('script');
			scripts.forEach((script) => {
				const newScript = document.createElement('script');
				if (script.src) {
					newScript.src = script.src;
				} else {
					newScript.textContent = '(function(){' + script.textContent + '})();';
				}
				script.parentNode.replaceChild(newScript, script);
			});
		}
	}

	render() {
		const { loading, html } = this.state;

		if (loading) {
			return el(
				'div',
				{ className: 'affbox-block-preview' },
				el(
					'div',
					{ className: 'affbox-block-preview-content', style: { padding: '40px', textAlign: 'center' } },
					el(Spinner),
					el('p', { style: { marginTop: '10px', color: '#666' } }, __('Loading preview...', 'affbox'))
				)
			);
		}

		return el(
			'div',
			{
				className: 'affbox-block-preview',
				style: this.props.isSyncing ? { opacity: 0.5, filter: 'grayscale(1)', pointerEvents: 'none', transition: 'opacity 150ms ease' } : undefined
			},
			html ? el(
				'div',
				{
					className: 'affbox-block-preview-content',
					ref: this.previewRef
				}
			) : el(
				'div',
				{ className: 'affbox-block-preview-content', style: { padding: '20px', background: '#f5f5f5', border: '1px solid #ddd', borderRadius: '4px' } },
				el('p', { style: { margin: 0, color: '#666', textAlign: 'center' } }, __('No content available. The product may need to be synced.', 'affbox'))
			)
		);
	}
}

registerBlockType('affbox/affiliate-product', {
	title: __('AffiliateBoxes', 'affbox'),
	icon: AffBoxIcon,
	category: 'widgets',
	description: __('Display an affiliate product from your Affiliate Boxes account.', 'affbox'),

	attributes: {
		slug: {
			type: 'string',
			default: ''
				},
				className: {
					type: 'string',
					default: ''
						}
						},

						edit: withSelect((select, props) => {
								const { getEntityRecords } = select('core');
								const { slug } = props.attributes;

								const query = {
									per_page: 100, // REST API max is 100; fetch first 100 products
									orderby: 'title',
									order: 'asc',
									status: 'publish',
									_fields: 'id,title,meta'
							};
								const products = getEntityRecords('postType', 'affiliate_product', query) || [];

								const selectedProduct = slug && products ? products.find(function (product) {
									const productSlug = product.meta && product.meta._affprod_slug ? product.meta._affprod_slug : '';
									return productSlug === slug;
								}) : null;

		return {
			products: products,
			selectedProduct: selectedProduct,
			isLoading: !select('core/data').hasFinishedResolution('core', 'getEntityRecords', ['postType', 'affiliate_product', query])
							};
						})(function ({ attributes, setAttributes, products, selectedProduct, isLoading }) {
							const { slug } = attributes;
							const [refreshKey, setRefreshKey] = useState(0);
							const [isSyncing, setIsSyncing] = useState(false);
							const [lastUpdated, setLastUpdated] = useState('');

							const productOptions = [
							{ value: '', label: __('— Select a product —', 'affbox') }
							];

							if (products && products.length > 0) {
								products.forEach(function (product) {
										const productSlug = product.meta && product.meta._affprod_slug ? product.meta._affprod_slug : '';
										const productTitle = product.title && product.title.rendered ? product.title.rendered : (product.title && product.title.raw ? product.title.raw : 'Product #' + product.id);

									if (productSlug) {
										productOptions.push({
											value: productSlug,
											label: productTitle
										});
									}
								});
							}

							const onSelectProduct = function (selectedSlug) {
								setAttributes({ slug: selectedSlug });
							};

							const blockProps = useBlockProps({
								className: 'affbox-block-editor'
							});

							const inspectorControls = el(
							InspectorControls,
							{},
							el(
							PanelBody,
							{ title: __('Affiliate Box Settings', 'affbox'), initialOpen: true },
							el(ChoicesSelect, {
								label: __('Select Affiliate Box', 'affbox'),
								value: slug,
								options: productOptions,
								onChange: onSelectProduct
								})
							)
							);

							const toolbarControls = el(
							BlockControls,
							{},
							el(
							ToolbarGroup,
							{},
							el(ToolbarButton, {
								icon: 'update',
								title: isSyncing ? __('Refreshing...', 'affbox') : __('Refresh product', 'affbox'),
								disabled: !slug || isSyncing,
								onClick: function () {
									if (!slug) {
										return; }
									setIsSyncing(true);
									wp.apiFetch({
										path: '/affbox/v1/refresh',
										method: 'POST',
										data: { slug: slug }
									}).then(function (resp) {
										if (resp && resp.last_updated) {
											setLastUpdated(resp.last_updated); }
										setRefreshKey(function (v) {
											return v + 1; });
										try {
											wp.data.dispatch('core/notices').createNotice('success', __('Product refreshed', 'affbox')); } catch (e) {
											}
									}).catch(function () {
										try {
											wp.data.dispatch('core/notices').createNotice('error', __('Failed to refresh product', 'affbox')); } catch (e) {
											}
									}).finally(function () {
										setIsSyncing(false); });
								}
							}),
							el(ToolbarButton, {
								icon: 'external',
								title: __('Edit in AffiliateBoxes.com', 'affbox'),
								disabled: !slug,
								onClick: function () {
									if (!slug) {
										return; }
									var itemType = selectedProduct && selectedProduct.meta && selectedProduct.meta._affprod_type ? selectedProduct.meta._affprod_type : '';
									var section = itemType === 'table' ? 'tables' : 'products';
									window.open('https://affiliateboxes.com/' + section + '/' + slug + '/', '_blank');
								}
							})
							)
							);

							if (isLoading) {
								return el(
								wp.element.Fragment,
								{},
								inspectorControls,
								el(
								'div',
								blockProps,
								el(
								Placeholder,
								{
									icon: AffBoxIcon,
									label: __('AffiliateBoxes', 'affbox'),
									instructions: __('Loading products...', 'affbox')
									},
								el(Spinner)
								)
								)
								);
							}

							let blockContent;

							if (!slug) {
								blockContent = el(
								Placeholder,
								{
									icon: AffBoxIcon,
									label: __('AffiliateBoxes', 'affbox'),
									instructions: __('Select a product from the block settings to display it here.', 'affbox')
								}
								);
							} else if (slug && !selectedProduct) {
								blockContent = el(
								'div',
								{ className: 'affbox-block-placeholder' },
								el('p', {}, __('Product not found. Please select another product or sync your products from the settings.', 'affbox'))
								);
							} else {
								const productTitle = selectedProduct.title && selectedProduct.title.rendered ? selectedProduct.title.rendered : (selectedProduct.title && selectedProduct.title.raw ? selectedProduct.title.raw : 'Product');

								blockContent = el(ProductPreview, {
									key: slug,
									slug: slug,
									title: productTitle,
									refreshKey: refreshKey,
									onPreviewLoaded: function (updatedAt) {
										setLastUpdated(updatedAt || ''); },
									isSyncing: isSyncing
								});
							}

							return el(
							wp.element.Fragment,
							{},
							toolbarControls,
							inspectorControls,
							el('div', blockProps, blockContent)
							);
						}),

						save: function () {
							return null;
						}
						});
