/**
 * Admin functions for AffBox.
 *
 * @package affbox
 */

const { __ } = wp.i18n || { __: (text) => text };

// Simple sprintf implementation for JavaScript
function sprintf(format, ...args) {
	let i = 0;
	return format.replace(/%[sd]/g, () => args[i++]);
}


// Function to get URL parameter
function getUrlParameter(name) {
	name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
	var regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
	var results = regex.exec(location.search);
	return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
};


//TODO replace with sending queue from the connection script
jQuery(document).ready(function ($) {
	if (getUrlParameter('sync') === 'true') {
		history.replaceState({}, '', window.location.href.replace(/sync=true&?/, ''));
		setTimeout(function () {
			const $syncButton = $('#sync-products');
			if ($syncButton.length && !$syncButton.prop('disabled')) {
				$syncButton.trigger('click');
			}
		}, 500);
	}

	// Background queue status polling
	let queuePollInterval = null;
	let emptyQueueCount = 0;
	let previousCompletedCount = 0;
	let syncTriggered = false;

	function showLoader() {
		$('#affbox-loader').show();
	}

	function hideLoader() {
		$('#affbox-loader').hide();
	}

	function startQueuePolling() {
		// Poll immediately
		pollQueueStatus();

		// Then poll every 5 seconds
		if (!queuePollInterval) {
			queuePollInterval = setInterval(pollQueueStatus, 10000);
		}
	}

	function refreshAvailableProducts() {
		$.ajax({
			url: affbox_ajax.ajax_url,
			type: 'POST',
			data: {
				action: 'affbox_get_available_products',
				nonce: affbox_ajax.nonce
			},
			dataType: 'json',
			success: function (response) {
				if (response.success && response.data && response.data.html) {
					$('#affbox-available-products').html(response.data.html);
				}
			}
		});
	}

	function pollQueueStatus() {
		$.ajax({
			url: affbox_ajax.ajax_url,
			type: 'POST',
			data: {
				action: 'affbox_get_queue_status',
				nonce: affbox_ajax.nonce
			},
			dataType: 'json',
			success: function (response) {
				if (response.success && response.data) {
					const status = response.data;
					const $container = $('#queue-status-container');
					const $status = $('#queue-status');

					// Check if completed count has changed (tasks finished between polls)
					const completedCount = status.total_completed || 0;
					const completedCountChanged = completedCount !== previousCompletedCount;

					if (completedCountChanged) {
						// Tasks completed since last poll - refresh products list
						refreshAvailableProducts();
					}

					// Update previous completed count
					previousCompletedCount = completedCount;

					if (status.has_pending && status.total_pending > 0) {
						// Reset empty counter
						emptyQueueCount = 0;
						syncTriggered = true;

						// Show queue status
						let message = `<i class="bi bi-arrow-repeat bi-spin"></i> `;
						message += sprintf(
							__('%d actions queued for processing', 'affbox'),
							status.total_pending
						);

						$status.html(message);
						$container.show();
					} else {
						// Queue is empty - only show "Sync complete" if a sync was actually in progress
						if (syncTriggered && emptyQueueCount === 0) {
							$status.html(`<i class="bi bi-check-lg text-success"></i> ${__('Sync complete', 'affbox')}`);
							$container.show();
							hideLoader();

							// Re-enable sync button
							$('#sync-products').prop('disabled', false).removeClass('disabled');

							refreshAvailableProducts();
						}

						emptyQueueCount++;

						// Hide after being empty for 2 polls (10 seconds)
						// if (emptyQueueCount >= 2) {
						//     $container.hide();

						//     // Stop polling after being empty for a while
						//     if (emptyQueueCount >= 12) { // 60 seconds
						//         clearInterval(queuePollInterval);
						//         queuePollInterval = null;
						//     }
						// }
					}
				}
			}
		});
	}

	// Start polling on page load if connected
	if ($('#sync-products').length) {
		startQueuePolling();
	}

	// Get translations from localized script with fallbacks
	const i18n = {
		syncing: __('Syncing...', 'affbox'),
		synced: __('Synced successfully!', 'affbox'),
		error: __('Error occurred. Please try again.', 'affbox'),
		confirm_disconnect: __('Are you sure you want to disconnect your site? This will remove the API key and you will need to reconnect.', 'affbox'),
		disconnecting: __('Disconnecting...', 'affbox'),
		last_synced: __('Last synchronized: ', 'affbox'),
		...(typeof affbox_ajax !== 'undefined' && affbox_ajax.i18n ? affbox_ajax.i18n : {})
	};

	$('#connect-site').on('click', function (e) {
		e.preventDefault();

		const $button = $(this);
		$button.prop('disabled', true);

		// Get the secret first
		$.ajax({
			url: affbox_ajax.ajax_url,
			type: 'POST',
			data: {
				action: 'affbox_start_connection',
				nonce: affbox_ajax.nonce
			},
			dataType: 'json',
			success: function (response) {
				if (response.success && response.data && response.data.secret) {
					var form = $('<form>', {
						'method': 'POST',
						'action': 'https://affiliateboxes.com/connect/'
					});
					form.append($('<input>', {
						'type': 'hidden',
						'name': 'site_name',
						'value': affbox_ajax.site_name
					}));
					form.append($('<input>', {
						'type': 'hidden',
						'name': 'site_url',
						'value': affbox_ajax.site_url
					}));
					form.append($('<input>', {
						'type': 'hidden',
						'name': 'secret',
						'value': response.data.secret
					}));
					$('body').append(form);
					form.submit();
					form.remove();
				} else {
					alert(i18n.error);
				}
			},
			error: function () {
				alert(i18n.error);
			},
			complete: function () {
				$button.prop('disabled', false);
			}
		});
	});

	$('#sync-products').on('click', function (e) {
		e.preventDefault();

		var $button = $(this);

		$button.prop('disabled', true).addClass('disabled');
		showLoader();
		syncTriggered = true;

		// Show immediate feedback
		const $queueStatus = $('#queue-status');
		$queueStatus.html(`<i class="bi bi-arrow-repeat bi-spin"></i> ${__('Starting manual sync...', 'affbox')}`);
		$('#queue-status-container').show();

		$.ajax({
			url: affbox_ajax.ajax_url,
			type: 'POST',
			data: {
				action: 'affbox_sync_products',
				nonce: affbox_ajax.nonce
			},
			dataType: 'json',
			success: function (response) {
				if (response.success) {
					// Update status message
					$queueStatus.html(`<i class="bi bi-arrow-repeat bi-spin"></i> ${response.data.message || __('Sync triggered...', 'affbox')}`);

					// Wait 2 seconds before checking status
					setTimeout(() => {
						// Reset empty queue counter
						emptyQueueCount = 0;

						// Trigger immediate poll
						pollQueueStatus();

						// Restart polling if it was stopped
						if (!queuePollInterval) {
							queuePollInterval = setInterval(pollQueueStatus, 5000);
						}
					}, 2000);

				} else {
					let errorMessage = response.data?.error || i18n.error;
					if (typeof response.data === "string") {
						errorMessage = response.data;
					} else {
						errorMessage = response.data?.error || i18n.error;
						if (response.data?.details) {
							errorMessage += `<br>${response.data.details}`;
						}
					}

					// Show error in queue status instead
					const $queueStatus = $('#queue-status');
					$queueStatus.html(`<i class="bi bi-exclamation-triangle text-warning"></i> ${errorMessage}`);
					$('#queue-status-container').show();
					hideLoader();

					$button.prop('disabled', false).removeClass('disabled');
				}
			},
			error: (xhr) => {
				let errorMessage = i18n.error;
				try {
					const json = xhr.responseJSON;
					if (json?.error) {
						errorMessage = json.error;
						if (json.details) {
							errorMessage += `<br>${json.details}`;
						}
					}
				} catch (e) {
				}

				// Show error in queue status instead
				const $queueStatus = $('#queue-status');
				$queueStatus.html(`<i class="bi bi-exclamation-triangle text-warning"></i> ${errorMessage}`);
				$('#queue-status-container').show();
				hideLoader();

				$button.prop('disabled', false).removeClass('disabled');
			}
		});
	});

	$(document).on('click', '#disconnect-site', function (e) {
		e.preventDefault();

		if (!confirm(i18n.confirm_disconnect)) {
			return;
		}

		const $button = $(this);
		const $status = $('#disconnect-status');

		$button.prop('disabled', true).addClass('disabled');
		$status.html(`<i class="bi bi-arrow-repeat bi-spin"></i> ${i18n.disconnecting}`);

		$.ajax({
			url: affbox_ajax.ajax_url,
			type: 'POST',
			data: {
				action: 'affbox_disconnect_site',
				nonce: affbox_ajax.nonce
			},
			dataType: 'json',
			success: (response) => {
				if (response.success) {
					$status.html(`<i class="bi bi-check-lg text-success"></i> ${response.data?.message || i18n.synced
					}`);
					setTimeout(() => window.location.reload(), 1500);
				} else {
					const errorMsg = response.error || i18n.error;
					$status.html(`<i class="bi bi-x-lg text-danger"></i> ${errorMsg}`);
					$button.prop('disabled', false).removeClass('disabled');
				}
			},
			error: (xhr) => {
				$status.html(`<i class="bi bi-x-lg text-danger"></i> ${i18n.error}`);
				$button.prop('disabled', false).removeClass('disabled');
			}
		});
	});

	$(document).on('click', '.copy-shortcode', function (e) {
		e.preventDefault();

		const shortcode = $(this).data('shortcode');
		const $button = $(this);
		const originalText = $button.text();
		const copyFallback = function () {
			const $temp = $('<input>');
			$('body').append($temp);
			$temp.val(shortcode).select();
			let ok = false;
			try {
				ok = document.execCommand('copy'); } catch (_) {
				}
				$temp.remove();
				return ok;
		};
		const after = function (success) {
			$button.text(success ? __('Copied!', 'affbox') : __('Press Ctrl+C to copy', 'affbox'));
			setTimeout(() => $button.text(originalText), 2000);
		};
		if (navigator.clipboard && navigator.clipboard.writeText) {
			navigator.clipboard.writeText(shortcode).then(() => after(true)).catch(() => after(copyFallback()));
		} else {
			after(copyFallback());
		}
	});

	// Click-to-copy on shortcode cell with tooltip and inline feedback
	$(document).on('click keydown', '.affbox-copy-shortcode', function (e) {
		if (e.type === 'keydown' && e.key !== 'Enter' && e.key !== ' ') {
			return;
		}
		e.preventDefault();
		const el = $(this);
		const shortcode = el.data('shortcode');
		const originalTitle = el.attr('title') || '';
		const $code = el.find('code');
		const restoreTarget = $code.length ? $code : el;
		const originalCodeHtml = restoreTarget.html();
		const fallbackCopy = function () {
			const $temp = $('<textarea style="position:absolute;left:-9999px;top:-9999px;">');
			$('body').append($temp);
			$temp.val(shortcode).select();
			let ok = false;
			try {
				ok = document.execCommand('copy'); } catch (_) {
				}
				$temp.remove();
				return ok;
		};
		const after = function (success) {
			el.attr('title', success ? __('Copied!', 'affbox') : __('Press Ctrl+C to copy', 'affbox'));
			restoreTarget.text(success ? __('Copied to clipboard', 'affbox') : __('Press Ctrl+C to copy', 'affbox'));
			// Force native title tooltip refresh by briefly removing and re-adding focus
			el.blur();
			setTimeout(() => {
				el.focus();
				setTimeout(() => {
					el.attr('title', originalTitle);
					restoreTarget.html(originalCodeHtml);
				}, 1500);
			}, 0);
		};
		if (navigator.clipboard && navigator.clipboard.writeText) {
			navigator.clipboard.writeText(shortcode).then(() => after(true)).catch(() => after(fallbackCopy()));
		} else {
			after(fallbackCopy());
		}
	});
});
