<?php
/**
 * Admin logic for the plugin.
 *
 * @package AffBox
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class AffBox_Admin
 */
class AffBox_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @var string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @var string $version The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of the plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_styles( $hook ) {
		if ( false === strpos( $hook, 'affbox' ) ) {
			return;
		}

		wp_enqueue_style(
			'affbox-inter-font',
			AFFBOX_PLUGIN_URL . 'admin/vendor/inter/inter.css',
			array(),
			$this->version
		);

		wp_enqueue_style(
			'affbox-bootstrap-icons',
			AFFBOX_PLUGIN_URL . 'admin/vendor/bootstrap-icons/bootstrap-icons.min.css',
			array(),
			$this->version
		);

		wp_enqueue_style(
			$this->plugin_name,
			AFFBOX_PLUGIN_URL . 'admin/css/affbox-admin.css',
			array( 'affbox-inter-font', 'affbox-bootstrap-icons' ),
			$this->version,
			'all'
		);
	}

	/**
	 * Register the block editor assets.
	 */
	public function enqueue_block_editor_assets() {
		// Enqueue Choices.js.
		wp_enqueue_script(
			'choices-js',
			AFFBOX_PLUGIN_URL . 'admin/js/choices.min.js',
			array(),
			'10.2.0',
			true
		);
		wp_enqueue_style(
			'choices-css',
			AFFBOX_PLUGIN_URL . 'admin/css/choices.min.css',
			array(),
			'10.2.0'
		);

		wp_enqueue_style(
			$this->plugin_name . '-editor',
			AFFBOX_PLUGIN_URL . 'admin/css/affbox-block-editor.css',
			array(),
			$this->version,
			'all'
		);
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		if ( false === strpos( $hook, 'affbox' ) ) {
			return;
		}

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		wp_enqueue_script(
			'affbox-bootstrap-js',
			AFFBOX_PLUGIN_URL . 'admin/vendor/bootstrap.bundle' . $suffix . '.js',
			array(),
			$this->version,
			true
		);

		wp_enqueue_script(
			$this->plugin_name,
			AFFBOX_PLUGIN_URL . 'admin/js/affbox-admin.js',
			array( 'jquery', 'wp-i18n', 'affbox-bootstrap-js' ),
			$this->version,
			true
		);

		// Set up translations.
		if ( function_exists( 'wp_set_script_translations' ) ) {
			wp_set_script_translations( $this->plugin_name, 'affbox' );
		}

		// Localize the script with data.
		wp_localize_script(
			$this->plugin_name,
			'affbox_ajax',
			array(
				'ajax_url'  => admin_url( 'admin-ajax.php' ),
				'nonce'     => wp_create_nonce( 'affbox_nonce' ),
				'site_name' => get_bloginfo( 'name' ),
				'site_url'  => get_site_url(),
				// These will now be handled by wp.i18n in JS.
				'i18n'      => array(
					'syncing'            => __( 'Syncing...', 'affbox' ),
					'synced'             => __( 'Synced successfully!', 'affbox' ),
					'error'              => __( 'Error occurred. Please try again.', 'affbox' ),
					'confirm_disconnect' => __( 'Are you sure you want to disconnect your site? This will remove the API key and you will need to reconnect.', 'affbox' ),
					'disconnecting'      => __( 'Disconnecting...', 'affbox' ),
					'last_synced'        => __( 'Last synchronized: ', 'affbox' ),
				),
			)
		);
	}

	/**
	 * Handle disconnecting the site by removing the API key.
	 *
	 * @throws Exception If settings update fails.
	 */
	public function disconnect_site() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		// Get current settings.
		$settings = get_option( 'affbox_settings', array() );

		// Check if API key exists.
		if ( empty( $settings['api_key'] ) ) {
			$error = 'No API key found to disconnect.';
			AffBox_Logger::log( 'Disconnect error: ' . $error );
			wp_send_json_error( $error, 400 );
		}

		try {
			// Remove the API key.
			$settings['api_key'] = '';
			$updated             = update_option( 'affbox_settings', $settings );

			if ( false === $updated ) {
				throw new Exception( 'Failed to update settings.' );
			}

			wp_send_json_success(
				array(
					'message' => __( 'Successfully disconnected. Refreshing...', 'affbox' ),
				)
			);

		} catch ( Exception $e ) {
			$error = 'Error disconnecting: ' . $e->getMessage();
			AffBox_Logger::log( 'Disconnect error: ' . $error );
			wp_send_json_error( $error, 500 );
		}
	}

	/**
	 * Add the admin menu for specified pages.
	 */
	public function add_plugin_admin_menu() {
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
		$svg_icon = 'data:image/svg+xml;base64,' . base64_encode(
			'<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="20" height="20">
<path d="M0 0 C1.32 0 2.64 0 4 0 C4 0.66 4 1.32 4 2 C7.63 2 11.26 2 15 2 C15 7.28 15 12.56 15 18 C9.39 18 3.78 18 -2 18 C-2.04125 15.40125 -2.0825 12.8025 -2.125 10.125 C-2.14320801 9.30789551 -2.16141602 8.49079102 -2.18017578 7.64892578 C-2.19477217 5.76504951 -2.09943482 3.88130688 -2 2 C-1.34 1.34 -0.68 0.68 0 0 Z M4 7 C4 8.98 4 10.96 4 13 C5.65 13 7.3 13 9 13 C9 11.02 9 9.04 9 7 C7.35 7 5.7 7 4 7 Z " fill="#3FBEC7" transform="translate(3,0)"/>
<path d="M0 0 C1.98 0 3.96 0 6 0 C6 3.3 6 6.6 6 10 C0.39 10 -5.22 10 -11 10 C-11 8.35 -11 6.7 -11 5 C-7.37 5 -3.74 5 0 5 C0 3.35 0 1.7 0 0 Z " fill="#FEC91E" transform="translate(12,8)"/>
<path d="M0 0 C1.98 0 3.96 0 6 0 C6 0.66 6 1.32 6 2 C4.68 2 3.36 2 2 2 C1.67 3.65 1.34 5.3 1 7 C-1.31 6.67 -3.62 6.34 -6 6 C-6.33 7.32 -6.66 8.64 -7 10 C-8.32 10 -9.64 10 -11 10 C-11 8.35 -11 6.7 -11 5 C-7.37 5 -3.74 5 0 5 C0 3.35 0 1.7 0 0 Z " fill="#F39238" transform="translate(12,8)"/>
<path d="M0 0 C1.98 0 3.96 0 6 0 C6 1.98 6 3.96 6 6 C4.02 6 2.04 6 0 6 C0 4.02 0 2.04 0 0 Z " fill="#D082FB" transform="translate(1,7)"/>
<path d="M0 0 C1.65 0 3.3 0 5 0 C5 1.98 5 3.96 5 6 C3.02 6 1.04 6 -1 6 C-0.67 4.02 -0.34 2.04 0 0 Z " fill="#E5CDF5" transform="translate(13,2)"/>
<path d="M0 0 C1.32 0 2.64 0 4 0 C3.67 1.98 3.34 3.96 3 6 C1.35 6.33 -0.3 6.66 -2 7 C-2.125 4.625 -2.125 4.625 -2 2 C-1.34 1.34 -0.68 0.68 0 0 Z " fill="#118BB6" transform="translate(3,0)"/>
<path d="M0 0 C1.98 0 3.96 0 6 0 C6 1.32 6 2.64 6 4 C3.69 4 1.38 4 -1 4 C-0.67 2.68 -0.34 1.36 0 0 Z " fill="#FDB213" transform="translate(6,14)"/>
<path d="M0 0 C1.98 0 3.96 0 6 0 C5.34 1.65 4.68 3.3 4 5 C2.68 5 1.36 5 0 5 C0 3.35 0 1.7 0 0 Z " fill="#C33643" transform="translate(1,13)"/>
<path d="M0 0 C0.99 0 1.98 0 3 0 C3 1.65 3 3.3 3 5 C1.68 5 0.36 5 -1 5 C-0.67 3.35 -0.34 1.7 0 0 Z " fill="#91C7D1" transform="translate(10,2)"/>
</svg>'
		);

		add_menu_page(
			__( 'Affiliate Boxes', 'affbox' ),
			__( 'Affiliate Boxes', 'affbox' ),
			'manage_options',
			'affbox',
			array( $this, 'display_plugin_admin_page' ),
			$svg_icon,
			76
		);
	}

	/**
	 * Display the main plugin admin page.
	 */
	public function display_plugin_admin_page() {
		include_once AFFBOX_PLUGIN_DIR . 'admin/partials/affbox-admin-display.php';
	}

	/**
	 * Sync products from affiliateboxes.com.
	 */
	public function sync_products() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		$current_settings = get_option( 'affbox_settings' );
		$api_key          = $current_settings['api_key'] ?? '';

		if ( empty( $api_key ) ) {
			wp_send_json_error( __( 'API key is not set', 'affbox' ) );
		}

		// Trigger sync on remote API.
		$response = wp_remote_get(
			'https://affiliateboxes.com/api/',
			array(
				'headers' => array(
					'Authorization' => 'Bearer ' . $api_key,
					'Accept'        => 'application/json',
				),
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response->get_error_message() );
		}

		$body          = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $body, true );

		// Check if the response has the expected structure.
		if ( empty( $response_data ) || ! is_array( $response_data ) || ! isset( $response_data['success'] ) || true !== $response_data['success'] ) {
			wp_send_json_error( __( 'Invalid API response format', 'affbox' ) );
		}

		// Return success immediately. The sync is handled by the remote server pushing data to the /wp-json/affbox/v1/receive endpoint.
		wp_send_json_success(
			array(
				'message' => $response_data['message'] ?? __( 'Sync triggered successfully', 'affbox' ),
			)
		);
	}

	/**
	 * Get sync queue status for polling.
	 */
	public function get_sync_status() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		$batch_id = isset( $_POST['batch_id'] ) ? sanitize_text_field( wp_unslash( $_POST['batch_id'] ) ) : '';

		if ( empty( $batch_id ) ) {
			wp_send_json_error( __( 'Batch ID is required', 'affbox' ) );
		}

		// Force run pending actions to ensure status is up to date.
		AffBox_Action_Scheduler::force_run_queue();

		$status = AffBox_Action_Scheduler::get_sync_queue_status( $batch_id );

		if ( isset( $status['error'] ) ) {
			wp_send_json_error( $status['error'] );
		}

		wp_send_json_success( $status );
	}

	/**
	 * Get overall queue status (not batch-specific).
	 */
	public function get_queue_status() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		// Force run pending actions to ensure status is up to date.
		AffBox_Action_Scheduler::force_run_queue();

		$status = AffBox_Action_Scheduler::get_overall_queue_status();

		wp_send_json_success( $status );
	}

	/**
	 * Get available products HTML.
	 */
	public function get_available_products() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		ob_start();
		include AFFBOX_PLUGIN_DIR . 'admin/partials/affbox-admin-products-list.php';
		$html = ob_get_clean();

		wp_send_json_success( array( 'html' => $html ) );
	}

	/**
	 * Start the connection process by generating a secret.
	 */
	public function start_connection() {
		check_ajax_referer( 'affbox_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'affbox' ) );
		}

		// Generate a random secret.
		$secret = wp_generate_password( 32, false );

		// Store it in a transient for 10 minutes.
		set_transient( 'affbox_connection_secret', $secret, 10 * MINUTE_IN_SECONDS );

		wp_send_json_success(
			array(
				'secret' => $secret,
			)
		);
	}
}
